<#
 This script creates the Azure AD applications needed for this sample and updates the configuration files
 for the visual Studio projects from the data in the Azure AD applications.

 Before running this script you need to install the AzureAD cmdlets as an administrator. 
 For this:
 1) Run Powershell as an administrator
 2) in the PowerShell window, type: Install-Module AzureAD

 There are four ways to run this script. For more information, read the AppCreationScripts.md file in the same folder as this script.
#>

#$CommandName: RegApplication, GetAllApplication
#$Test: 0 means not test, 1 means test.
[CmdletBinding()]
Param(
#[Parameter(Mandatory=$True)]
[string]$AdminUserName, 
#[Parameter(Mandatory=$True)]
[string]$AdminPassword, 
#[Parameter(Mandatory=$True)]
[string]$CommandName, 
#[Parameter(Mandatory=$False)]
[string]$AppTitle, 
[string]$IsBase64,
[switch]$Test)

[Console]::OutputEncoding = [System.Text.Encoding]::UTF8
$PSDefaultParameterValues['*:Encoding'] = 'utf8'

$global:psFilePath = $MyInvocation.MyCommand.Definition
$global:adminname = $AdminUserName
$global:adminpsw = $AdminPassword
$global:commandn = $CommandName
$global:appName = $AppTitle
$global:istest = $Test
$global:isbase64 = $IsBase64

$global:LogFilePath = ""
$global:IsConnectAzure = $False
$global:CommandAppRegister="RegApplication"
$global:CommandGetAllApplication="GetAllApplication"
$global:CommandIsAppInList="AppIsInList"

$global:RegexForEmail = "^(?("")("".+?(?<!\\)""@)|(([0-9a-z]((\.(?!\.))|[-!#\$%&'\*\+/=\?\^`\{\}\|~\w])*)(?<=[0-9a-z])@))(?(\[)(\[(\d{1,3}\.){3}\d{1,3}\])|(([0-9a-z][-\w]*[0-9a-z]*\.)+[a-z0-9][\-a-z0-9]{0,22}[a-z0-9]))$"
# $txt = "simple@example.com"
# $re = [regex]::IsMatch($txt, $global:RegexForEmail, "IgnoreCase")
# $txt = "simple@example.com"
# $re = [regex]::IsMatch($txt, $global:RegexForEmail, "IgnoreCase")
# $txt = "very.common@example.com"
# $re = [regex]::IsMatch($txt, $global:RegexForEmail, "IgnoreCase")
# $txt = "disposable.style.email.with+symbol@example.com"
# $re = [regex]::IsMatch($txt, $global:RegexForEmail, "IgnoreCase")
# $txt = "other.email-with-hyphen@example.com"
# $re = [regex]::IsMatch($txt, $global:RegexForEmail, "IgnoreCase")
# $txt = "fully-qualified-domain@example.com"
# $re = [regex]::IsMatch($txt, $global:RegexForEmail, "IgnoreCase")
# $txt = "user.name+tag+sorting@example.com" 
# $re = [regex]::IsMatch($txt, $global:RegexForEmail, "IgnoreCase")
# $txt = "x@example.com"
# $re = [regex]::IsMatch($txt, $global:RegexForEmail, "IgnoreCase")
# $txt = "example-indeed@strange-example.com"
# $re = [regex]::IsMatch($txt, $global:RegexForEmail, "IgnoreCase")
# $txt = "admin@mailserver1" #it's false
# $re = [regex]::IsMatch($txt, $global:RegexForEmail, "IgnoreCase")
# $txt = "example@s.example"
# $re = [regex]::IsMatch($txt, $global:RegexForEmail, "IgnoreCase")
# $txt = """ ""@example.org"
# $re = [regex]::IsMatch($txt, $global:RegexForEmail, "IgnoreCase")
# $txt = """john..doe""@example.org"
# $re = [regex]::IsMatch($txt, $global:RegexForEmail, "IgnoreCase")

# $txt = "Abc.example.com"
# $re = [regex]::IsMatch($txt, $global:RegexForEmail, "IgnoreCase")
# $txt = "A@b@c@example.com"
# $re = [regex]::IsMatch($txt, $global:RegexForEmail, "IgnoreCase")
# $txt = "a""b(c)d,e:f;g<h>i[j\k]l@example.com"
# $re = [regex]::IsMatch($txt, $global:RegexForEmail, "IgnoreCase")
# $txt = "just""not""right@example.com"
# $re = [regex]::IsMatch($txt, $global:RegexForEmail, "IgnoreCase")
# $txt = "this is""not\allowed@example.com"
# $re = [regex]::IsMatch($txt, $global:RegexForEmail, "IgnoreCase")
# $txt = "this\ still\""not\\allowed@example.com"
# $re = [regex]::IsMatch($txt, $global:RegexForEmail, "IgnoreCase")
# $txt = "1234567890123456789012345678901234567890123456789012345678901234+x@example.com"
# $re = [regex]::IsMatch($txt, $global:RegexForEmail, "IgnoreCase")
# $txt = "john..doe@example.com"
# $re = [regex]::IsMatch($txt, $global:RegexForEmail, "IgnoreCase")
# $txt = "john.doe@example..com"
# $re = [regex]::IsMatch($txt, $global:RegexForEmail, "IgnoreCase")


Function Decodebase64([string]$Base64String){
    $temp = [System.Convert]::FromBase64String($Base64String)
    return  [System.Text.Encoding]::UTF8.GetString($temp)
}

if((-Not $global:adminname) -and (-Not $global:adminpsw) -and (-Not $global:commandn) -and (-Not $global:appName ) -and (-Not $global:istest)) {
    $global:istest = $True
    $global:commandn = $global:CommandAppRegister
} else {
    if ($global:isbase64 -eq 1){
        $global:adminname = Decodebase64 -Base64String $global:adminname
        $global:adminpsw = Decodebase64 -Base64String $global:adminpsw
        $global:appName = Decodebase64 -Base64String $global:appName
    }
}

if($global:istest) {
    Write-Host $global:adminname
    Write-Host $global:appName
    Write-Host $global:commandn
    Write-Host $global:istest
}


# $Arg1=""
# $Arg2=""
# $Arg3=""
# $Arg4=""
# $Arg5=""
# $Arg6=""
# $Arg7=""
# $Arg8=""
# $ArgLength=0

$global:ErrorDetail=""
$global:ErrorNotFoundTenantName = "OneDrive_Not_Found_Tenant"
$global:ErrorCanntConnectAzure = "OneDrive_Can_Not_Connect_Azure"
$global:ErrorInvalidUserNameAndPsw = "OneDrive_Invalid_UserName_Password"
$global:ErrorCanntGetAzureDetail = "OneDrive_Can_Not_Get_Azure_Detail"
$global:ErrorCanntNewApplication = "OneDrive_Can_Not_New_Application"
$global:ErrorCanntSetApplicationResouceAccess = "OneDrive_Can_Not_Set_Application_Access"
$global:ErrorProgramError = "OneDrive_Program_Error"
$global:ErrorCanntFindApplication = "OneDrive_Can_Not_Find_Application"
$global:ErrorAppNameTooLong = "OneDrive_Application_Name_Too_Long"
$global:ErrorInstallAzureModule = "OneDrive_Application_Not_Install_Azure_AD_Module"
$global:ErrorMFAEnabled = "OneDrive_MFA_ENABLED"

# Function ResetArgs
# {
#     $ArgLength=0
# }

# Function SetArgs([Int]$argsLength, [string]$arg1, [string]$arg2,[string]$arg3,[string]$arg4,[string]$arg5,[string]$arg6,[string]$arg7,[string]$arg8)
# {
# 	$Arg1=$arg1
# 	$Arg2=$arg2
# 	$Arg3=$arg3
# 	$Arg4=$arg4
# 	$Arg5=$arg5
# 	$Arg6=$arg6
# 	$Arg7=$arg7
# 	$Arg8=$arg8
# 	$ArgLength=$argsLength
# }

Function GetExceptionMessage([string]$scriptStackTrace, [int]$linenumber, [string]$exstacktrace, [string]$message) {
    $DebugMessage="Message:"+$message+"        LineNumber:"+$linenumber+"        StackTrace:"+$scriptStackTrace+"        StackTrace:"+ $exstacktrace+"         Detail:"+$global:ErrorDetail
    return $DebugMessage
}

# Create a password that can be used as an application key
Function ComputePassword
{
    $aesManaged = New-Object "System.Security.Cryptography.AesManaged"  -ErrorAction Stop
    $aesManaged.Mode = [System.Security.Cryptography.CipherMode]::CBC
    $aesManaged.Padding = [System.Security.Cryptography.PaddingMode]::Zeros
    $aesManaged.BlockSize = 128
    $aesManaged.KeySize = 256
    $aesManaged.GenerateKey()
    return [System.Convert]::ToBase64String($aesManaged.Key)
}

Function CreateLogFile(){
    $enginePath = (get-item $global:psFilePath).Directory.Parent.Parent.FullName
    $engineLogPath = Join-Path $enginePath "Logs"
    $engineLogOffice365Path = Join-Path $engineLogPath "Office365Log"
    if (-Not (Test-Path -Path $engineLogOffice365Path)) {
        New-Item -ItemType Directory -Path $engineLogOffice365Path
    }
    $timeStr = (Get-Date).ToString("yyyyMMdd_HHmmss")
    $fileName = 'OD_PS_'+$timeStr+'.log'
    $temp = Join-Path $engineLogOffice365Path $fileName
    return $temp
}

Function WriteLog($message)
{
    $timeStr = (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
    $writeMessage= $timeStr +'    ' + $message
    try {
        $writeMessage | Add-Content $global:LogFilePath -ErrorAction Stop -Force
    }
    catch{
        #$errorMessage = GetExceptionMessage -linenumber $PSItem.InvocationInfo.ScriptLineNumber -scriptStackTrace $PSItem.ScriptStackTrace -exstacktrace $PSItem.Exception.StackTrace -message $PSItem.Exception.Message
        #WriteLog("End Add Content for application "+$AADAppTitle+". failed:"+ $errorMessage)
    }
}

# Create an application key
# See https://www.sabin.io/blog/adding-an-azure-active-directory-application-and-key-using-powershell/
Function CreateAppKey([DateTime] $fromDate, [double] $durationInYears, [string]$pw)
{
    $endDate = $fromDate.AddYears($durationInYears) 
    $keyId = [guid]::newguid().ToString();
    $key = New-Object Microsoft.Open.AzureAD.Model.PasswordCredential -ErrorAction Stop
    $key.StartDate = $fromDate
    $key.EndDate = $endDate
    $key.Value = $pw
    $key.KeyId = $keyId
    return $key
}

# Adds the requiredAccesses (expressed as a pipe separated string) to the requiredAccess structure
# The exposed permissions are in the $exposedPermissions collection, and the type of permission (Scope | Role) is 
# described in $permissionType
Function AddResourcePermission($requiredAccess, `
                               $exposedPermissions, [string]$requiredAccesses, [string]$permissionType)
{
        foreach($permission in $requiredAccesses.Trim().Split("|"))
        {
            foreach($exposedPermission in $exposedPermissions)
            {
                if ($exposedPermission.Value -eq $permission)
                {
                    $resourceAccess = New-Object Microsoft.Open.AzureAD.Model.ResourceAccess -ErrorAction Stop
                    $resourceAccess.Type = $permissionType # Scope = Delegated permissions | Role = Application permissions
                    $resourceAccess.Id = $exposedPermission.Id # Read directory data
                    $requiredAccess.ResourceAccess.Add($resourceAccess)
                }
            }
        }
}

#
# Exemple: GetRequiredPermissions "Microsoft Graph"  "Graph.Read|User.Read"
# See also: http://stackoverflow.com/questions/42164581/how-to-configure-a-new-azure-ad-application-through-powershell
Function GetRequiredPermissions([string] $applicationID, [string] $requiredDelegatedPermissions, [string]$requiredApplicationPermissions, $servicePrincipal)
{
    # If we are passed the service principal we use it directly, otherwise we find it from the display name (which might not be unique)
    WriteLog("Start Get required permission")
    if ($servicePrincipal)
    {
        $sp = $servicePrincipal
    }
    else
    {
		try{
			$sp = Get-AzureADServicePrincipal -Filter "AppId eq '$applicationID'" -ErrorAction Stop
		}
		catch{
            $errorMessage = GetExceptionMessage -linenumber $PSItem.InvocationInfo.ScriptLineNumber -scriptStackTrace $PSItem.ScriptStackTrace -exstacktrace $PSItem.Exception.StackTrace -message $PSItem.Exception.Message
            WriteLog("End Get required permission. failed:"+ $errorMessage)
            $global:ErrorDetail=$PSItem.ToString()
            throw $ErrorCanntSetApplicationResouceAccess
		}
    }
    $appid = $sp.AppId
    $requiredAccess = New-Object Microsoft.Open.AzureAD.Model.RequiredResourceAccess -ErrorAction Stop
    $requiredAccess.ResourceAppId = $appid 
    $requiredAccess.ResourceAccess = New-Object System.Collections.Generic.List[Microsoft.Open.AzureAD.Model.ResourceAccess] -ErrorAction Stop

    # $sp.Oauth2Permissions | Select Id,AdminConsentDisplayName,Value: To see the list of all the Delegated permissions for the application:
    if ($requiredDelegatedPermissions)
    {
        AddResourcePermission $requiredAccess -exposedPermissions $sp.Oauth2Permissions -requiredAccesses $requiredDelegatedPermissions -permissionType "Scope"
    }
    
    # $sp.AppRoles | Select Id,AdminConsentDisplayName,Value: To see the list of all the Application permissions for the application
    if ($requiredApplicationPermissions)
    {
        AddResourcePermission $requiredAccess -exposedPermissions $sp.AppRoles -requiredAccesses $requiredApplicationPermissions -permissionType "Role"
    }
    WriteLog("End Get required permission.")
    return $requiredAccess
}

Function GetRequiredPermissions_SP([string] $applicationDisplayName, [string] $requiredDelegatedPermissions, [string]$requiredApplicationPermissions, $servicePrincipal)
{
    # If we are passed the service principal we use it directly, otherwise we find it from the display name (which might not be unique)
    WriteLog("Start Get required permission for sharepoint")
    if ($servicePrincipal)
    {
        $sp = $servicePrincipal
    }
    else
    {
		
		try{
			
			$sp = Get-AzureADServicePrincipal -All $true | ? { $_.DisplayName -match "Office 365 SharePoint Online" }

		}
		catch{
            $errorMessage = GetExceptionMessage -linenumber $PSItem.InvocationInfo.ScriptLineNumber -scriptStackTrace $PSItem.ScriptStackTrace -exstacktrace $PSItem.Exception.StackTrace -message $PSItem.Exception.Message
            WriteLog("End Get required permission. failed:"+ $errorMessage)
            $global:ErrorDetail=$PSItem.ToString()
            throw $ErrorCanntSetApplicationResouceAccess
		}
    }
		
	$appid = $sp.AppId
	WriteLog("App id for SP:" + $appid)
    $requiredAccess = New-Object Microsoft.Open.AzureAD.Model.RequiredResourceAccess -ErrorAction Stop
    $requiredAccess.ResourceAppId = $appid 
    $requiredAccess.ResourceAccess = New-Object System.Collections.Generic.List[Microsoft.Open.AzureAD.Model.ResourceAccess] -ErrorAction Stop

    if ($requiredDelegatedPermissions)
    {
        AddResourcePermission $requiredAccess -exposedPermissions $sp.Oauth2Permissions -requiredAccesses $requiredDelegatedPermissions -permissionType "Scope"
		WriteLog("Perm1 :" + $requiredDelegatedPermissions)
    }
	
    if ($requiredApplicationPermissions)
    {
        AddResourcePermission $requiredAccess -exposedPermissions $sp.AppRoles -requiredAccesses $requiredApplicationPermissions -permissionType "Role"
		WriteLog("Perm2:" + $requiredApplicationPermissions)
    }
    WriteLog("End Get required permission for sharepoint.")
    return $requiredAccess
}


# Replace the value of an appsettings of a given key in an XML App.Config file.
Function ReplaceSetting([string] $configFilePath, [string] $key, [string] $newValue)
{
    [xml] $content = Get-Content $configFilePath
    $appSettings = $content.configuration.appSettings; 
    $keyValuePair = $appSettings.SelectSingleNode("descendant::add[@key='$key']")
    if ($keyValuePair)
    {
        $keyValuePair.value = $newValue;
    }
    else
    {
        Throw "Key '$key' not found in file '$configFilePath'"
    }
	$content.save($configFilePath)
}

Function GetAzureName
{
    # $tenantId is the Active Directory Tenant. This is a GUID which represents the "Directory ID" of the AzureAD tenant
    # into which you want to create the apps. Look it up in the Azure portal in the "Properties" of the Azure AD.

    # Login to Azure PowerShell (interactive if credentials are not already provided:
    # you'll need to sign-in with creds enabling your to create apps in the tenant)
    [CmdletBinding()]
    param(
        [PSCredential] $Credential,
        [Parameter(HelpMessage='Tenant ID (This is a GUID which represents the "Directory ID" of the AzureAD tenant into which you want to create the apps')]
        [string] $tenantId
    )
    
    WriteLog("Start Connect Azure AD.")
    try{
        if (!$Credential -and $tenantId)
        {
            $creds = Connect-AzureAD -TenantId $tenantId -ErrorAction Stop
        }
        else
        {
            if (!$tenantId)
            {
                $creds = Connect-AzureAD -Credential $Credential -ErrorAction Stop
            }
            else
            {
                $creds = Connect-AzureAD -TenantId $tenantId -Credential $Credential -ErrorAction Stop
            }
        }
        WriteLog("End Connect Azure AD. TenantName:"+$creds.Tenant.Id)
        $global:IsConnectAzure = $True
    }
    catch
    {
        $errorMessage = GetExceptionMessage -linenumber $PSItem.InvocationInfo.ScriptLineNumber -scriptStackTrace $PSItem.ScriptStackTrace -exstacktrace $PSItem.Exception.StackTrace -message $PSItem.Exception.Message
        WriteLog("End Connect Azure AD. failed:"+ $errorMessage)
        $isMatch = $PSItem.Exception.Message -match 'Invalid username or password'
        if ($isMatch) {
            $global:ErrorDetail=$PSItem.ToString()
            throw $ErrorInvalidUserNameAndPsw
        }
        $isMatch = $PSItem.Exception.Message -match 'ID3242'
        if ($isMatch) {
            $global:ErrorDetail=$PSItem.ToString()
            throw $ErrorInvalidUserNameAndPsw
        }
        $isMatch = $PSItem.Exception.Message -match 'The user account does not exist'
        if ($isMatch) {
            $global:ErrorDetail=$PSItem.ToString()
            throw $ErrorInvalidUserNameAndPsw
        }
        $isMatch = $PSItem.Exception.Message -match 'Unknown User Type'
        if ($isMatch) {
            $global:ErrorDetail=$PSItem.ToString()
            throw $ErrorInvalidUserNameAndPsw
        }

        $isMatch = $PSItem.Exception.Message -match "The term 'Connect-AzureAD' is not recognized as the name of a cmdlet"
        if ($isMatch){
            $global:ErrorDetail = $PSItem.ToString()
            throw $ErrorInstallAzureModule
        }

        $isMatch = $PSItem.Exception.Message -match "AADSTS50076"
        if ($isMatch){
            $global:ErrorDetail = $PSItem.ToString()
            throw $ErrorMFAEnabled
        }

        throw $ErrorCanntConnectAzure
    }

    if (!$tenantId)
    {
        $tenantId = $creds.Tenant.Id
    }

    WriteLog("Start Get Azure Tenant Name.")
    try{
        $tenant = Get-AzureADTenantDetail -ErrorAction Stop
        $tenantName =  ($tenant.VerifiedDomains | Where { $_._Default -eq $True }).Name
        WriteLog("End Get Azure Tenant Name:["+$tenantName+"]")
    }
    catch{
        $errorMessage = GetExceptionMessage -linenumber $PSItem.InvocationInfo.ScriptLineNumber -scriptStackTrace $PSItem.ScriptStackTrace -exstacktrace $PSItem.Exception.StackTrace -message $PSItem.Exception.Message
        WriteLog("End Get Azure Tenant Name. failed:"+ $errorMessage)
        throw $ErrorCanntGetAzureDetail
    }

    return $tenantName
}

Function BuildReturnResult([bool]$isFind, [string]$appId,[string]$objectId, [string]$appName, [string]$appPassword, [string]$tenantName)
{
    $isFindInt = 0
    if ($isFind) {
        $isFindInt = 1
    }

    if($appPassword) {
        $url = "https://portal.azure.com/#blade/Microsoft_AAD_IAM/ApplicationBlade/appId/$appId/objectId/$objectId"
    } else {
        $url = ""
    }

    $appInformationObj = [pscustomobject]@{
        AppId = $appId
        Status = 0
        IsFind = $isFindInt
        ObjectId = $objectId
        Password = $appPassword
        AppTitle = $appName
        Url = $url
        TenantName = $tenantName
    }
    return ConvertTo-Json -InputObject $appInformationObj -Compress
    <#
    if($appPassword) {
        $url = "https://portal.azure.com/#blade/Microsoft_AAD_IAM/ApplicationBlade/appId/$appId/objectId/$objectId"
        $appInformation = '{"AppId":"'+$appId+'", "Status":0, "IsFind":'+$isFindInt+',"ObjectId":"'+$objectId+'","Password":"'+$appPassword+'","AppTitle":"'+$appName+'", "Url":"'+$url+'", "TenantName":"'+$tenantName+'"}'
        return $appInformation
    } else {
        $appInformation = '{"AppId":"'+$appId+'","Status":0, "IsFind":'+$isFindInt+',"ObjectId":"'+$objectId+'","AppTitle":"'+$appName+'", "TenantName":"'+$tenantName+'"}'
        return $appInformation
    }
    #>
}

Function BuildErrorReturnResult([string]$messageId)
{
    $buildresult = '{"Status":1,"MessageId":"'+$messageId+'"}'
    #$buildresult = '{"Status":1,"MessageId":"'+$messageId+'","ErrorDetail":"'+$global:ErrorDetail+'"}'
    return $buildresult
}

Function ConfigureApplications
{
<#.Description
   This function creates the Azure AD applications for the sample in the provided Azure AD tenant and updates the
   configuration files in the client and service project  of the visual studio solution (App.Config and Web.Config)
   so that they are consistent with the Applications parameters
#> 
    [CmdletBinding()]
    param(
        [PSCredential] $Credential,
        [Parameter(HelpMessage='Tenant ID (This is a GUID which represents the "Directory ID" of the AzureAD tenant into which you want to create the apps')]
        [string] $tenantId,
		[string] $AADAppTitle
    )

   process
   {
    WriteLog("Start Register Application ["+$AADAppTitle+"]")
    $tenantName = GetAzureName -Credential $Credential -tenantId $TenantId -ErrorAction Stop

    try{
        WriteLog("Start Find Application "+$AADAppTitle)
        $filterfs = "DisplayName eq "
        $filterfs+= "'"
        $filterfs+=  ($AADAppTitle -replace "'","`'`'")
        $filterfs+= "'"


        $CurrentApp =Get-AzureADApplication -Filter $filterfs -ErrorAction Stop
        if($CurrentApp) {
            WriteLog("End Find Application "+$AADAppTitle + ", found.")
            # if($CurrentApp.RequiredResourceAccess) {
            #     WriteLog("Start Find Application "+$AADAppTitle + " resource access.")
            #     foreach($resource in $CurrentApp.RequiredResourceAccess) {
            #         if ($resource.ResourceAppId -eq '00000003-0000-0000-c000-000000000000') {
            #             WriteLog("End Find Application "+$AADAppTitle + " resource access. found")
            #             $findresult = BuildReturnResult -isFind 1 -appId $CurrentApp.AppId -appName $CurrentApp.DisplayName -objectId $CurrentApp.ObjectId
            #             return $findresult
            #         }
            #     }
               
            # }
            # WriteLog("End Find Application "+$AADAppTitle + " resource access. not found, will delete the application first and register application again.")
            WriteLog("Remove Application "+$AADAppTitle + " start.")
            Remove-AzureADApplication -ObjectId $CurrentApp.ObjectId
            $findAfterRemove = 1
            $findCount = 0
            do {
                Start-Sleep -Seconds 5
                $CurrentApp =Get-AzureADApplication -Filter $filterfs -ErrorAction Stop
                if($CurrentApp) {
                    $findAfterRemove = 1
                } else {
                    $findAfterRemove = 0
                    break
                }
                $findCount = $findCount + 1
            }
            while($findCount -lt 30)

            if ($findAfterRemove -eq 1) {
                throw "Remove application failed."
            }
            WriteLog("Remove Application "+$AADAppTitle + " end.")
        } else {
            WriteLog("End Find Application "+$AADAppTitle + ", not found. will register application")
        }
    }
    catch {
        $errorMessage = GetExceptionMessage -linenumber $PSItem.InvocationInfo.ScriptLineNumber -scriptStackTrace $PSItem.ScriptStackTrace -exstacktrace $PSItem.Exception.StackTrace -message $PSItem.Exception.Message
        WriteLog("End Find Application "+$AADAppTitle+". failed:"+ $errorMessage)
        throw $ErrorCanntFindApplication
    }
    
    # Get a 2 years application key for the client Application
    $pw = ComputePassword
    
    $fromDate = [DateTime]::UtcNow;
    #$fromDate = $fromDate.AddMonths(-1);
    WriteLog("Create Application "+$AADAppTitle + " secret key")
    $key = CreateAppKey -fromDate $fromDate -durationInYears 50 -pw $pw -ErrorAction Stop

    #get all application information
    #$allApplication = Get-AzureADApplication -All $true 

    try{
        WriteLog("Start New Application "+$AADAppTitle)
        $identityUrl = "https://" + $tenantName +"/"+[guid]::newguid().ToString()
        
                                                  #-IdentifierUris $identityUrl `
        $clientAadApplication = New-AzureADApplication -DisplayName $AADAppTitle `
                                                  -PasswordCredentials $key `
                                                  -IdentifierUris $identityUrl `
                                                  -PublicClient $False -ErrorAction Stop
        WriteLog("End New Application "+$AADAppTitle)
    }
    catch{
        $errorMessage = GetExceptionMessage -linenumber $PSItem.InvocationInfo.ScriptLineNumber -scriptStackTrace $PSItem.ScriptStackTrace -exstacktrace $PSItem.Exception.StackTrace -message $PSItem.Exception.Message
        WriteLog("End New Application "+$AADAppTitle+". failed:"+ $errorMessage)
        $global:ErrorDetail=$PSItem.ToString()
        throw $ErrorCanntNewApplication
    }

    $currentAppId = $clientAadApplication.AppId

    try{
        WriteLog("Start New Application "+$AADAppTitle + " principal")
        $principal = New-AzureADServicePrincipal -AppId $currentAppId -Tags {WindowsAzureActiveDirectoryIntegratedApp} -ErrorAction Stop
        WriteLog("End New Application "+$AADAppTitle + " principal")
    }
    catch{
        $errorMessage = GetExceptionMessage -linenumber $PSItem.InvocationInfo.ScriptLineNumber -scriptStackTrace $PSItem.ScriptStackTrace -exstacktrace $PSItem.Exception.StackTrace -message $PSItem.Exception.Message
        WriteLog("End New Application "+$AADAppTitle+" principal. failed:"+ $errorMessage)
        $global:ErrorDetail = $PSItem.ToString()
        throw $ErrorCanntSetApplicationResouceAccess
    }
    #WriteLog( "Done.")
    #WriteLog( "AppID:$currentAppId")

    # URL of the AAD application in the Azure portal
    $clientPortalUrl = "https://portal.azure.com/#@"+$tenantName+"/blade/Microsoft_AAD_IAM/ApplicationBlade/appId/"+$clientAadApplication.AppId+"/objectId/"+$clientAadApplication.ObjectId
    
    try{
        WriteLog("Start Add Content for application "+$AADAppTitle)
        try {
            Add-Content -Value "<tr><td>client</td><td>$currentAppId</td><td><a href='$clientPortalUrl'>$AADAppTitle</a></td></tr>" -Path createdApps.html -ErrorAction Stop
        }
        catch {
            $errorMessage = GetExceptionMessage -linenumber $PSItem.InvocationInfo.ScriptLineNumber -scriptStackTrace $PSItem.ScriptStackTrace -exstacktrace $PSItem.Exception.StackTrace -message $PSItem.Exception.Message
            WriteLog("End Add Content for application "+$AADAppTitle+". failed:"+ $errorMessage)
        }
        WriteLog("End Add Content for application "+$AADAppTitle)
    }
    catch{
        $errorMessage = GetExceptionMessage -linenumber $PSItem.InvocationInfo.ScriptLineNumber -scriptStackTrace $PSItem.ScriptStackTrace -exstacktrace $PSItem.Exception.StackTrace -message $PSItem.Exception.Message
        WriteLog("End Add Content for application "+$AADAppTitle+". failed:"+ $errorMessage)
        $global:ErrorDetail = $PSItem.ToString()
        throw $ErrorCanntSetApplicationResouceAccess
    }

    $requiredResourcesAccess = New-Object System.Collections.Generic.List[Microsoft.Open.AzureAD.Model.RequiredResourceAccess] -ErrorAction Stop
    # Add Required Resources Access (from 'client' to 'service')
    #WriteLog( "Getting access from 'client' to 'service'")
    #$requiredPermissions = GetRequiredPermissions -applicationDisplayName "TodoListService" `
    #                                              -requiredDelegatedPermissions "user_impersonation";

    $requiredAdminRolePermissions = "People.Read.All|Sites.FullControl.All|User.Read.All|Group.Read.All|Calendars.Read|Contacts.Read|Directory.Read.All|Files.ReadWrite.All|Mail.Read|Notes.Read.All"
	
	
    $requiredPermissions = GetRequiredPermissions -applicationID "00000003-0000-0000-c000-000000000000" `
                                                    -requiredDelegatedPermissions "" -requiredApplicationPermissions $requiredAdminRolePermissions  -ErrorAction Stop
                                                    
    $requiredResourcesAccess.Add($requiredPermissions)
	
    
  
	$requiredAdminRoleAppPermissionsForSharePoint = "Sites.FullControl.All|User.ReadWrite.All"
    	$requiredAdminRoleDelePermissionsForSharePoint = "AllSites.FullControl"
	
	$url = "Office 365 SharePoint Online"
    	$requiredPermissionsForSharePoint = GetRequiredPermissions_SP -applicationDisplayName $url `
                                                   -requiredDelegatedPermissions $requiredAdminRoleDelePermissionsForSharePoint -requiredApplicationPermissions $requiredAdminRoleAppPermissionsForSharePoint  -ErrorAction Stop


	try{
        WriteLog("Start Set Application "+$AADAppTitle + " Resource access.")
        Set-AzureADApplication -ObjectId $clientAadApplication.ObjectId -RequiredResourceAccess $requiredPermissions,$requiredPermissionsForSharePoint -ErrorAction Stop
       
        WriteLog("End Set Application "+$AADAppTitle + " Resource access.")
	}
	catch {
        $errorMessage = GetExceptionMessage -linenumber $PSItem.InvocationInfo.ScriptLineNumber -scriptStackTrace $PSItem.ScriptStackTrace -exstacktrace $PSItem.Exception.StackTrace -message $PSItem.Exception.Message
        WriteLog("End Set Application "+$AADAppTitle + " Resource access. failed:" + $errorMessage)
        $global:ErrorDetail=$PSItem.ToString()
        throw $ErrorCanntSetApplicationResouceAccess
    }
    $correctresult = BuildReturnResult -isFind 0 -appId $clientAadApplication.AppId -appName $clientAadApplication.DisplayName -objectId $clientAadApplication.ObjectId -appPassword $pw -tenantName $tenantName
   
   
    #$roleCompany = Get-AzureADDirectoryRole -Filter "DisplayName eq 'Global Administrator'"
	#Add-AzureADDirectoryRoleMember -ObjectId $roleCompany.objectId  -RefObjectId $principal.objectId -ErrorAction Stop
   
    return $correctresult
  }
}


Function GetApplications{
    [CmdletBinding()]
    param(
        [PSCredential] $Credential,
        [Parameter(HelpMessage='Tenant ID (This is a GUID which represents the "Directory ID" of the AzureAD tenant into which you want to create the apps')]
        [string] $tenantId,
		[string] $AADAppTitle
    )
    process
    {
    
        $tenantName = GetAzureName -Credential $Credential -tenantId $TenantId -ErrorAction Stop
        if(!$tenantName){
            return
        }
        try{
            $allApplication = Get-AzureADApplication -All $true  -ErrorAction Stop
            foreach($app in $allApplication){
                WriteAppInformationWithoutPassword -currentAppId $app.AppId -objectId $app.ObjectId -AADAppTitle $app.DisplayName $tenantName
            }
            return
        }
        catch{
            $result = OutputException("Can't new azure app")
            if($result)
            {
                return
            }
        }
    }
}

 

Function IsApplicationIsList{
    [CmdletBinding()]
    param(
        [PSCredential] $Credential,
        [Parameter(HelpMessage='Tenant ID (This is a GUID which represents the "Directory ID" of the AzureAD tenant into which you want to create the apps')]
        [string] $tenantId,
		[string] $AADAppTitle
    )
    process
    { 
        $tenantName = GetAzureName -Credential $Credential -tenantId $tenantId -ErrorAction Stop
        if(!$tenantName){
			
            throw "Not Find TenantName"
        }
        try{
         #"DisplayName eq 'irui0017'"
            $filterfs = "DisplayName eq"
            $filterfs+= "'"
            $filterfs+=  $AADAppTitle
            $filterfs+= "'"

    
            $CurrentApp =Get-AzureADApplication -Filter $filterfs -ErrorAction Stop
            if($CurrentApp)
            {
                 WriteAppInformationWithoutPassword -currentAppId $CurrentApp.AppId -objectId $CurrentApp.ObjectId -AADAppTitle $CurrentApp.DisplayName $tenantName
                 return "Find:$AADAppTitle"
            
            }
            else
            {
                 return "NoFind: $AADAppTitle"
   
            }
        }
        catch{
            $result = OutputException("Can't new azure app")
            if($result)
            { 
                  return "Exception: $AADAppTitle"
            }
        }
    }
}
try{
    $global:LogFilePath = CreateLogFile
    try {
        Set-Content -Value "<html><body><table>" -Path createdApps.html -ErrorAction Stop
        Add-Content -Value "<thead><tr><th>Application</th><th>AppId</th><th>Url in the Azure portal</th></tr></thead><tbody>" -Path createdApps.html -ErrorAction Stop
    }
    catch {
        $errorMessage = GetExceptionMessage -linenumber $PSItem.InvocationInfo.ScriptLineNumber -scriptStackTrace $PSItem.ScriptStackTrace -exstacktrace $PSItem.Exception.StackTrace -message $PSItem.Exception.Message
        WriteLog("End Add Content for application "+$AADAppTitle+". failed:"+ $errorMessage)
    }
    Set-ExecutionPolicy -Scope Process -ExecutionPolicy Unrestricted -Force -ErrorAction Stop
}
catch {
    $global:ErrorDetail=$PSItem.ToString()
    $result = BuildErrorReturnResult -messageId $ErrorProgramError
    #Write-Output $result
    return
}

if(!(($global:commandn -eq $global:CommandAppRegister) -or ($global:commandn -eq $global:CommandGetAllApplication) -or ($global:commandn -eq $global:CommandIsAppInList) )){
    $global:ErrorDetail = "Command name is not right"
    $result = BuildErrorReturnResult -messageId $ErrorProgramError
    Write-Output $result
    return
}


if([string]::IsNullOrEmpty($global:adminpsw)) {
    if($global:istest) {
        $global:adminpsw = "Please input password!"
    }
    Else {
        $global:ErrorDetail = "Please input password"
        $result = BuildErrorReturnResult -messageId $ErrorProgramError
        Write-Output $result
        return
    }

}
if([string]::IsNullOrEmpty($global:adminname)) {
    if($global:istest){
        $global:adminname="Please input userAccount"
    }
    Else
    {
        $global:ErrorDetail = "Please input user name"
        $result = BuildErrorReturnResult -messageId $ErrorProgramError
        Write-Output $result
        return
    }
}
if([string]::IsNullOrEmpty($global:appName)) {
    if($global:commandn -eq $global:CommandAppRegister){
        if($global:istest){
            $global:appName="LhyTestAppReg_"+[guid]::newguid().ToString();
        }
        Else{ 
            $global:ErrorDetail = "Please input the application title"
            $result = BuildErrorReturnResult -messageId $ErrorProgramError
            Write-Output $result
            return
        }
    }
} else {
    if($global:commandn -eq $global:CommandAppRegister) {
        if ($global:commandn.Length -gt 256) {
            $global:ErrorDetail = "Application name is too long, it must be less than 256"
            $result = BuildErrorReturnResult -messageId $ErrorAppNameTooLong
            Write-Output $result + " Count:" + $global:commandn.Length
            return
        }
    }
}

if (-Not [regex]::IsMatch($global:adminname , $global:RegexForEmail, "IgnoreCase")) {
    $global:ErrorDetail = "Invalid username and password"
    $result = BuildErrorReturnResult -messageId $global:ErrorInvalidUserNameAndPsw
    Write-Output $result
    return
}

$global:secpasswd = ConvertTo-SecureString $global:adminpsw -AsPlainText -Force -ErrorAction Stop
$global:seCredential = New-Object System.Management.Automation.PSCredential ($global:adminname, $global:secpasswd) -ErrorAction Stop
$global:tenantIdStr="" #"16379ed5-208f-4519-b441-48141a223a6c"
$global:TestTitle= $global:appName #"LhyTestAppReg"
#Install-Module AzureAD
# Run interactively (will ask you for the tenant ID)

if($global:commandn -eq $global:CommandAppRegister) {
    try{
        WriteLog($global:adminname)
        $returnResult = ConfigureApplications -Credential $global:seCredential -tenantId $global:tenantIdStr -AADAppTitle $global:TestTitle -ErrorAction Stop
        #WriteLog($returnResult)
        Write-Output $returnResult
        try{
            if ($global:IsConnectAzure -eq $True) {
                Disconnect-AzureAD
            }
            return
        }
        catch {
            $errorMessage = GetExceptionMessage -linenumber $PSItem.InvocationInfo.ScriptLineNumber -scriptStackTrace $PSItem.ScriptStackTrace -exstacktrace $PSItem.Exception.StackTrace -message $PSItem.Exception.Message
            WriteLog("Disconnect Azure failed:" + $errorMessage)
            return
        }
    }
    catch {
        $errorMessage = GetExceptionMessage -linenumber $PSItem.InvocationInfo.ScriptLineNumber -scriptStackTrace $PSItem.ScriptStackTrace -exstacktrace $PSItem.Exception.StackTrace -message $PSItem.Exception.Message
        WriteLog("End Register Application "+$AADAppTitle + ".failed:" + $errorMessage)
        $MessageId = $PSItem.Exception.Message
        $errorResult = BuildErrorReturnResult -messageId $MessageId
        Write-Output $errorResult
        return
    }
}

if($global:commandn -eq $global:CommandGetAllApplication){
    GetApplications -Credential $global:seCredential -tenantId $global:tenantIdStr -AADAppTitle $global:TestTitle -ErrorAction Stop
}

if($global:commandn -eq $global:CommandIsAppInList) {
    IsApplicationIsList -Credential $global:seCredential -tenantId $global:tenantIdStr -AADAppTitle $global:TestTitle -ErrorAction Stop
}

# Need open the URL: https://portal.azure.com/#blade/Microsoft_AAD_IAM/ApplicationBlade/appId/{0:AppID}/objectId/{1:ObjectID}
# SIG # Begin signature block
# MIInFAYJKoZIhvcNAQcCoIInBTCCJwECAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQUrR12bnTgczdPfniS+TqGoJJ8
# a3SggiDEMIIFjTCCBHWgAwIBAgIQDpsYjvnQLefv21DiCEAYWjANBgkqhkiG9w0B
# AQwFADBlMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMSQwIgYDVQQDExtEaWdpQ2VydCBBc3N1cmVk
# IElEIFJvb3QgQ0EwHhcNMjIwODAxMDAwMDAwWhcNMzExMTA5MjM1OTU5WjBiMQsw
# CQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cu
# ZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQw
# ggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQC/5pBzaN675F1KPDAiMGkz
# 7MKnJS7JIT3yithZwuEppz1Yq3aaza57G4QNxDAf8xukOBbrVsaXbR2rsnnyyhHS
# 5F/WBTxSD1Ifxp4VpX6+n6lXFllVcq9ok3DCsrp1mWpzMpTREEQQLt+C8weE5nQ7
# bXHiLQwb7iDVySAdYyktzuxeTsiT+CFhmzTrBcZe7FsavOvJz82sNEBfsXpm7nfI
# SKhmV1efVFiODCu3T6cw2Vbuyntd463JT17lNecxy9qTXtyOj4DatpGYQJB5w3jH
# trHEtWoYOAMQjdjUN6QuBX2I9YI+EJFwq1WCQTLX2wRzKm6RAXwhTNS8rhsDdV14
# Ztk6MUSaM0C/CNdaSaTC5qmgZ92kJ7yhTzm1EVgX9yRcRo9k98FpiHaYdj1ZXUJ2
# h4mXaXpI8OCiEhtmmnTK3kse5w5jrubU75KSOp493ADkRSWJtppEGSt+wJS00mFt
# 6zPZxd9LBADMfRyVw4/3IbKyEbe7f/LVjHAsQWCqsWMYRJUadmJ+9oCw++hkpjPR
# iQfhvbfmQ6QYuKZ3AeEPlAwhHbJUKSWJbOUOUlFHdL4mrLZBdd56rF+NP8m800ER
# ElvlEFDrMcXKchYiCd98THU/Y+whX8QgUWtvsauGi0/C1kVfnSD8oR7FwI+isX4K
# Jpn15GkvmB0t9dmpsh3lGwIDAQABo4IBOjCCATYwDwYDVR0TAQH/BAUwAwEB/zAd
# BgNVHQ4EFgQU7NfjgtJxXWRM3y5nP+e6mK4cD08wHwYDVR0jBBgwFoAUReuir/SS
# y4IxLVGLp6chnfNtyA8wDgYDVR0PAQH/BAQDAgGGMHkGCCsGAQUFBwEBBG0wazAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsGAQUFBzAC
# hjdodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURS
# b290Q0EuY3J0MEUGA1UdHwQ+MDwwOqA4oDaGNGh0dHA6Ly9jcmwzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcmwwEQYDVR0gBAowCDAGBgRV
# HSAAMA0GCSqGSIb3DQEBDAUAA4IBAQBwoL9DXFXnOF+go3QbPbYW1/e/Vwe9mqyh
# hyzshV6pGrsi+IcaaVQi7aSId229GhT0E0p6Ly23OO/0/4C5+KH38nLeJLxSA8hO
# 0Cre+i1Wz/n096wwepqLsl7Uz9FDRJtDIeuWcqFItJnLnU+nBgMTdydE1Od/6Fmo
# 8L8vC6bp8jQ87PcDx4eo0kxAGTVGamlUsLihVo7spNU96LHc/RzY9HdaXFSMb++h
# UD38dglohJ9vytsgjTVgHAIDyyCwrFigDkBjxZgiwbJZ9VVrzyerbHbObyMt9H5x
# aiNrIv8SuFQtJ37YOtnwtoeW/VvRXKwYw02fc7cBqZ9Xql4o4rmUMIIGrjCCBJag
# AwIBAgIQBzY3tyRUfNhHrP0oZipeWzANBgkqhkiG9w0BAQsFADBiMQswCQYDVQQG
# EwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNl
# cnQuY29tMSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwHhcNMjIw
# MzIzMDAwMDAwWhcNMzcwMzIyMjM1OTU5WjBjMQswCQYDVQQGEwJVUzEXMBUGA1UE
# ChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQg
# UlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMIICIjANBgkqhkiG9w0BAQEF
# AAOCAg8AMIICCgKCAgEAxoY1BkmzwT1ySVFVxyUDxPKRN6mXUaHW0oPRnkyibaCw
# zIP5WvYRoUQVQl+kiPNo+n3znIkLf50fng8zH1ATCyZzlm34V6gCff1DtITaEfFz
# sbPuK4CEiiIY3+vaPcQXf6sZKz5C3GeO6lE98NZW1OcoLevTsbV15x8GZY2UKdPZ
# 7Gnf2ZCHRgB720RBidx8ald68Dd5n12sy+iEZLRS8nZH92GDGd1ftFQLIWhuNyG7
# QKxfst5Kfc71ORJn7w6lY2zkpsUdzTYNXNXmG6jBZHRAp8ByxbpOH7G1WE15/teP
# c5OsLDnipUjW8LAxE6lXKZYnLvWHpo9OdhVVJnCYJn+gGkcgQ+NDY4B7dW4nJZCY
# OjgRs/b2nuY7W+yB3iIU2YIqx5K/oN7jPqJz+ucfWmyU8lKVEStYdEAoq3NDzt9K
# oRxrOMUp88qqlnNCaJ+2RrOdOqPVA+C/8KI8ykLcGEh/FDTP0kyr75s9/g64ZCr6
# dSgkQe1CvwWcZklSUPRR8zZJTYsg0ixXNXkrqPNFYLwjjVj33GHek/45wPmyMKVM
# 1+mYSlg+0wOI/rOP015LdhJRk8mMDDtbiiKowSYI+RQQEgN9XyO7ZONj4KbhPvbC
# dLI/Hgl27KtdRnXiYKNYCQEoAA6EVO7O6V3IXjASvUaetdN2udIOa5kM0jO0zbEC
# AwEAAaOCAV0wggFZMBIGA1UdEwEB/wQIMAYBAf8CAQAwHQYDVR0OBBYEFLoW2W1N
# hS9zKXaaL3WMaiCPnshvMB8GA1UdIwQYMBaAFOzX44LScV1kTN8uZz/nupiuHA9P
# MA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDCDB3BggrBgEFBQcB
# AQRrMGkwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBBBggr
# BgEFBQcwAoY1aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1
# c3RlZFJvb3RHNC5jcnQwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybDMuZGln
# aWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcmwwIAYDVR0gBBkwFzAI
# BgZngQwBBAIwCwYJYIZIAYb9bAcBMA0GCSqGSIb3DQEBCwUAA4ICAQB9WY7Ak7Zv
# mKlEIgF+ZtbYIULhsBguEE0TzzBTzr8Y+8dQXeJLKftwig2qKWn8acHPHQfpPmDI
# 2AvlXFvXbYf6hCAlNDFnzbYSlm/EUExiHQwIgqgWvalWzxVzjQEiJc6VaT9Hd/ty
# dBTX/6tPiix6q4XNQ1/tYLaqT5Fmniye4Iqs5f2MvGQmh2ySvZ180HAKfO+ovHVP
# ulr3qRCyXen/KFSJ8NWKcXZl2szwcqMj+sAngkSumScbqyQeJsG33irr9p6xeZmB
# o1aGqwpFyd/EjaDnmPv7pp1yr8THwcFqcdnGE4AJxLafzYeHJLtPo0m5d2aR8XKc
# 6UsCUqc3fpNTrDsdCEkPlM05et3/JWOZJyw9P2un8WbDQc1PtkCbISFA0LcTJM3c
# HXg65J6t5TRxktcma+Q4c6umAU+9Pzt4rUyt+8SVe+0KXzM5h0F4ejjpnOHdI/0d
# KNPH+ejxmF/7K9h+8kaddSweJywm228Vex4Ziza4k9Tm8heZWcpw8De/mADfIBZP
# J/tgZxahZrrdVcA6KYawmKAr7ZVBtzrVFZgxtGIJDwq9gdkT/r+k0fNX2bwE+oLe
# Mt8EifAAzV3C+dAjfwAL5HYCJtnwZXZCpimHCUcr5n8apIUP/JiW9lVUKx+A+sDy
# Divl1vupL0QVSucTDh3bNzgaoSv27dZ8/DCCBrAwggSYoAMCAQICEAitQLJg0pxM
# n17Nqb2TrtkwDQYJKoZIhvcNAQEMBQAwYjELMAkGA1UEBhMCVVMxFTATBgNVBAoT
# DERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEhMB8GA1UE
# AxMYRGlnaUNlcnQgVHJ1c3RlZCBSb290IEc0MB4XDTIxMDQyOTAwMDAwMFoXDTM2
# MDQyODIzNTk1OVowaTELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJ
# bmMuMUEwPwYDVQQDEzhEaWdpQ2VydCBUcnVzdGVkIEc0IENvZGUgU2lnbmluZyBS
# U0E0MDk2IFNIQTM4NCAyMDIxIENBMTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCC
# AgoCggIBANW0L0LQKK14t13VOVkbsYhC9TOM6z2Bl3DFu8SFJjCfpI5o2Fz16zQk
# B+FLT9N4Q/QX1x7a+dLVZxpSTw6hV/yImcGRzIEDPk1wJGSzjeIIfTR9TIBXEmtD
# mpnyxTsf8u/LR1oTpkyzASAl8xDTi7L7CPCK4J0JwGWn+piASTWHPVEZ6JAheEUu
# oZ8s4RjCGszF7pNJcEIyj/vG6hzzZWiRok1MghFIUmjeEL0UV13oGBNlxX+yT4Us
# SKRWhDXW+S6cqgAV0Tf+GgaUwnzI6hsy5srC9KejAw50pa85tqtgEuPo1rn3MeHc
# reQYoNjBI0dHs6EPbqOrbZgGgxu3amct0r1EGpIQgY+wOwnXx5syWsL/amBUi0nB
# k+3htFzgb+sm+YzVsvk4EObqzpH1vtP7b5NhNFy8k0UogzYqZihfsHPOiyYlBrKD
# 1Fz2FRlM7WLgXjPy6OjsCqewAyuRsjZ5vvetCB51pmXMu+NIUPN3kRr+21CiRshh
# WJj1fAIWPIMorTmG7NS3DVPQ+EfmdTCN7DCTdhSmW0tddGFNPxKRdt6/WMtyEClB
# 8NXFbSZ2aBFBE1ia3CYrAfSJTVnbeM+BSj5AR1/JgVBzhRAjIVlgimRUwcwhGug4
# GXxmHM14OEUwmU//Y09Mu6oNCFNBfFg9R7P6tuyMMgkCzGw8DFYRAgMBAAGjggFZ
# MIIBVTASBgNVHRMBAf8ECDAGAQH/AgEAMB0GA1UdDgQWBBRoN+Drtjv4XxGG+/5h
# ewiIZfROQjAfBgNVHSMEGDAWgBTs1+OC0nFdZEzfLmc/57qYrhwPTzAOBgNVHQ8B
# Af8EBAMCAYYwEwYDVR0lBAwwCgYIKwYBBQUHAwMwdwYIKwYBBQUHAQEEazBpMCQG
# CCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wQQYIKwYBBQUHMAKG
# NWh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRSb290
# RzQuY3J0MEMGA1UdHwQ8MDowOKA2oDSGMmh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNv
# bS9EaWdpQ2VydFRydXN0ZWRSb290RzQuY3JsMBwGA1UdIAQVMBMwBwYFZ4EMAQMw
# CAYGZ4EMAQQBMA0GCSqGSIb3DQEBDAUAA4ICAQA6I0Q9jQh27o+8OpnTVuACGqX4
# SDTzLLbmdGb3lHKxAMqvbDAnExKekESfS/2eo3wm1Te8Ol1IbZXVP0n0J7sWgUVQ
# /Zy9toXgdn43ccsi91qqkM/1k2rj6yDR1VB5iJqKisG2vaFIGH7c2IAaERkYzWGZ
# gVb2yeN258TkG19D+D6U/3Y5PZ7Umc9K3SjrXyahlVhI1Rr+1yc//ZDRdobdHLBg
# XPMNqO7giaG9OeE4Ttpuuzad++UhU1rDyulq8aI+20O4M8hPOBSSmfXdzlRt2V0C
# FB9AM3wD4pWywiF1c1LLRtjENByipUuNzW92NyyFPxrOJukYvpAHsEN/lYgggnDw
# zMrv/Sk1XB+JOFX3N4qLCaHLC+kxGv8uGVw5ceG+nKcKBtYmZ7eS5k5f3nqsSc8u
# pHSSrds8pJyGH+PBVhsrI/+PteqIe3Br5qC6/To/RabE6BaRUotBwEiES5ZNq0RA
# 443wFSjO7fEYVgcqLxDEDAhkPDOPriiMPMuPiAsNvzv0zh57ju+168u38HcT5uco
# P6wSrqUvImxB+YJcFWbMbA7KxYbD9iYzDAdLoNMHAmpqQDBISzSoUSC7rRuFCOJZ
# DW3KBVAr6kocnqX9oKcfBnTn8tZSkP2vhUgh+Vc7tJwD7YZF9LRhbr9o4iZghurI
# r6n+lB3nYxs6hlZ4TjCCBsAwggSooAMCAQICEAxNaXJLlPo8Kko9KQeAPVowDQYJ
# KoZIhvcNAQELBQAwYzELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJ
# bmMuMTswOQYDVQQDEzJEaWdpQ2VydCBUcnVzdGVkIEc0IFJTQTQwOTYgU0hBMjU2
# IFRpbWVTdGFtcGluZyBDQTAeFw0yMjA5MjEwMDAwMDBaFw0zMzExMjEyMzU5NTla
# MEYxCzAJBgNVBAYTAlVTMREwDwYDVQQKEwhEaWdpQ2VydDEkMCIGA1UEAxMbRGln
# aUNlcnQgVGltZXN0YW1wIDIwMjIgLSAyMIICIjANBgkqhkiG9w0BAQEFAAOCAg8A
# MIICCgKCAgEAz+ylJjrGqfJru43BDZrboegUhXQzGias0BxVHh42bbySVQxh9J0J
# dz0Vlggva2Sk/QaDFteRkjgcMQKW+3KxlzpVrzPsYYrppijbkGNcvYlT4DotjIdC
# riak5Lt4eLl6FuFWxsC6ZFO7KhbnUEi7iGkMiMbxvuAvfTuxylONQIMe58tySSge
# TIAehVbnhe3yYbyqOgd99qtu5Wbd4lz1L+2N1E2VhGjjgMtqedHSEJFGKes+JvK0
# jM1MuWbIu6pQOA3ljJRdGVq/9XtAbm8WqJqclUeGhXk+DF5mjBoKJL6cqtKctvdP
# bnjEKD+jHA9QBje6CNk1prUe2nhYHTno+EyREJZ+TeHdwq2lfvgtGx/sK0YYoxn2
# Off1wU9xLokDEaJLu5i/+k/kezbvBkTkVf826uV8MefzwlLE5hZ7Wn6lJXPbwGqZ
# IS1j5Vn1TS+QHye30qsU5Thmh1EIa/tTQznQZPpWz+D0CuYUbWR4u5j9lMNzIfMv
# wi4g14Gs0/EH1OG92V1LbjGUKYvmQaRllMBY5eUuKZCmt2Fk+tkgbBhRYLqmgQ8J
# JVPxvzvpqwcOagc5YhnJ1oV/E9mNec9ixezhe7nMZxMHmsF47caIyLBuMnnHC1mD
# jcbu9Sx8e47LZInxscS451NeX1XSfRkpWQNO+l3qRXMchH7XzuLUOncCAwEAAaOC
# AYswggGHMA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQM
# MAoGCCsGAQUFBwMIMCAGA1UdIAQZMBcwCAYGZ4EMAQQCMAsGCWCGSAGG/WwHATAf
# BgNVHSMEGDAWgBS6FtltTYUvcyl2mi91jGogj57IbzAdBgNVHQ4EFgQUYore0GH8
# jzEU7ZcLzT0qlBTfUpwwWgYDVR0fBFMwUTBPoE2gS4ZJaHR0cDovL2NybDMuZGln
# aWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0UlNBNDA5NlNIQTI1NlRpbWVTdGFt
# cGluZ0NBLmNybDCBkAYIKwYBBQUHAQEEgYMwgYAwJAYIKwYBBQUHMAGGGGh0dHA6
# Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBYBggrBgEFBQcwAoZMaHR0cDovL2NhY2VydHMu
# ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0UlNBNDA5NlNIQTI1NlRpbWVT
# dGFtcGluZ0NBLmNydDANBgkqhkiG9w0BAQsFAAOCAgEAVaoqGvNG83hXNzD8deNP
# 1oUj8fz5lTmbJeb3coqYw3fUZPwV+zbCSVEseIhjVQlGOQD8adTKmyn7oz/AyQCb
# Ex2wmIncePLNfIXNU52vYuJhZqMUKkWHSphCK1D8G7WeCDAJ+uQt1wmJefkJ5ojO
# fRu4aqKbwVNgCeijuJ3XrR8cuOyYQfD2DoD75P/fnRCn6wC6X0qPGjpStOq/CUkV
# NTZZmg9U0rIbf35eCa12VIp0bcrSBWcrduv/mLImlTgZiEQU5QpZomvnIj5EIdI/
# HMCb7XxIstiSDJFPPGaUr10CU+ue4p7k0x+GAWScAMLpWnR1DT3heYi/HAGXyRkj
# gNc2Wl+WFrFjDMZGQDvOXTXUWT5Dmhiuw8nLw/ubE19qtcfg8wXDWd8nYiveQclT
# uf80EGf2JjKYe/5cQpSBlIKdrAqLxksVStOYkEVgM4DgI974A6T2RUflzrgDQkfo
# QTZxd639ouiXdE4u2h4djFrIHprVwvDGIqhPm73YHJpRxC+a9l+nJ5e6li6FV8Bg
# 53hWf2rvwpWaSxECyIKcyRoFfLpxtU56mWz06J7UWpjIn7+NuxhcQ/XQKujiYu54
# BNu90ftbCqhwfvCXhHjjCANdRyxjqCU4lwHSPzra5eX25pvcfizM/xdMTQCi2NYB
# DriL7ubgclWJLCcZYfZ3AYwwggcFMIIE7aADAgECAhAOI7n57/d2ejf8eVRP3GwZ
# MA0GCSqGSIb3DQEBCwUAMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2Vy
# dCwgSW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25p
# bmcgUlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwHhcNMjIwNDExMDAwMDAwWhcNMjMw
# NDExMjM1OTU5WjCBiTELMAkGA1UEBhMCVVMxEjAQBgNVBAgTCU1pbm5lc290YTEV
# MBMGA1UEBxMMRWRlbiBQcmFpcmllMRswGQYDVQQKExJBcmNzZXJ2ZSAoVVNBKSBM
# TEMxFTATBgNVBAsTDEFyY3NlcnZlIExMQzEbMBkGA1UEAxMSQXJjc2VydmUgKFVT
# QSkgTExDMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEA52wFjxGF4so9
# Yeb5wXukWNfyw02I82HhI2eZHhxg3FL/pRmRXn3CPalA8HK7Sthe7udGte3XFjSE
# 4fnrzGQcERbmg5tbmzT0d3NP8zOlQF7pwOkljJDikMwJkpFfhIa9zd2u3NR+xZoz
# iL5dtuo1aaV28poMB1lFGqB03tJTZtihPLoj18BZztPxFwjc7iCpqq3mgcS4HhAq
# A/tatET7mXom4DBc/22AqCNnZkJG6VgvslyoUCY8wzZOH5hTPTcFMwNrzMaMi/G/
# D+Xk/xCCFM8bRZNeX6oT6JYymBQWiGJMlC5OQem1ILtai1TKy6KGb+Hn3DsfpPX0
# oemug+lD20K9+dsTcz8Estq+XU0GNFbtIRUtGEHWGHDFB3To/5hVyq7dK3L7t3ku
# gQbgIaYORJpFbwqZlTgTL3ZE55ZPU+MAgh7PnXe63xhVmYYbAXHGaLscJvGaLuy6
# mkV+dqqlk05LNbrcSnNGKNUysnpLhRilN+fJUxhv6FdrtwC8SF5NAgMBAAGjggIG
# MIICAjAfBgNVHSMEGDAWgBRoN+Drtjv4XxGG+/5hewiIZfROQjAdBgNVHQ4EFgQU
# AFLdEjyA1FrmGkd7PFRGUontIIMwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoG
# CCsGAQUFBwMDMIG1BgNVHR8Ega0wgaowU6BRoE+GTWh0dHA6Ly9jcmwzLmRpZ2lj
# ZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNENvZGVTaWduaW5nUlNBNDA5NlNIQTM4
# NDIwMjFDQTEuY3JsMFOgUaBPhk1odHRwOi8vY3JsNC5kaWdpY2VydC5jb20vRGln
# aUNlcnRUcnVzdGVkRzRDb2RlU2lnbmluZ1JTQTQwOTZTSEEzODQyMDIxQ0ExLmNy
# bDA+BgNVHSAENzA1MDMGBmeBDAEEATApMCcGCCsGAQUFBwIBFhtodHRwOi8vd3d3
# LmRpZ2ljZXJ0LmNvbS9DUFMwgZQGCCsGAQUFBwEBBIGHMIGEMCQGCCsGAQUFBzAB
# hhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wXAYIKwYBBQUHMAKGUGh0dHA6Ly9j
# YWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNENvZGVTaWduaW5n
# UlNBNDA5NlNIQTM4NDIwMjFDQTEuY3J0MAwGA1UdEwEB/wQCMAAwDQYJKoZIhvcN
# AQELBQADggIBAHErY7354bB/5xhVZOQwtIQXWsyYrPon3km65LyHRFa4OmluXjpq
# LC8aKqdcpb4QEciQoDhO9BikMWWb9IUC78zBCv/J4ZsDcHHM2tcEiYA/Jxk6I3R0
# BAMS0LnLqLWsF7noUPz/pEOYMPSLExQ1HT+fBbaHE9Vj6d5GiDiox/ALeLEJGjo2
# AvutHypIf7tB+shud4S6I0rE+bCtE7vrnU3BTpkoBT2w2uuJwJV7onXTB48cZ6pi
# y2j88z2dQ4appclAjysTzbx9BgLNZUdUDdIUUIPCP3wIUH4vxmjGx7DwFJ4sDLRU
# 9mM5XRjgjZKw0p8sDn7pAM8ZxxgDifBYfb4S+3W0AEL+1Iul1O+RAENvTS7+xZWo
# 8OVdSKvSnOGEKQBqP7JJFMOZ9wXLbqowQgxS08edOMMBWHYC+bkCso/Rs2LHeKP7
# 6F19OFa85BAjrtB2B2z5fYSEAhqU8+UQhUp1vfPTaO9+JDv1tru2FaFD3Ja4GDoV
# DYoh+TS0QWJELxihkr/VxwJkgmDM5owi68Uav54RmcaZZxpn8sErkrM+WC4Kt7Tt
# 4Q5Dm9f9T6Aj0XzqhHwCqMR/6cYIMaHnLHNOQcswX8L3b1zUzkQz2QulwoBAaefg
# FzWMz/nAkZrNcf9SWPnXVnknqMcHXkED5xIs3cDyJ1uQX0sgvthFFZzZMYIFujCC
# BbYCAQEwfTBpMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4x
# QTA/BgNVBAMTOERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5nIFJTQTQw
# OTYgU0hBMzg0IDIwMjEgQ0ExAhAOI7n57/d2ejf8eVRP3GwZMAkGBSsOAwIaBQCg
# cDAQBgorBgEEAYI3AgEMMQIwADAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAc
# BgorBgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAjBgkqhkiG9w0BCQQxFgQUGnN9
# xrok8ACGXYfzYNcSIlMqA5cwDQYJKoZIhvcNAQEBBQAEggGAeWtVzUfjzz5vHea+
# jY9J7a2j0rWZOfG7SKieELQWtPebvIFOtAWjdA8ZqeYOmIZXL2yxqRpRW3gktrKW
# plBsqOKQuaXevn1isoQ7kAY26p46Qfiyt6ysKpFPiG3S5w12+tLM7HvqM3dOnjuS
# poCQvWQvAw3UTXxUCmpaCcVh5FcrXZ7WbK4zQy5eUNS41eC3z6c9ALzoqw1d1Zba
# 4p0+h+zLKYM14vuOf3rVFDDlL9bcE5KzSOtloVjtzZQLrWN2n1X1SMFs3UP6UX/t
# iFRfb2pf/M/n+mLy2tsq+uTm5vi+tKqb5be6KynoRx3axSjXQ3le0DewJYXUU+S4
# Hxob098RocJ6JuUkZkJBjXl3a2d9VqVI9iWb0Qg+uKwwv0s4eXPARkQG7AWFt6s8
# 398HPnug86CgSTW+ZFRtLHGMNa32sgHNFlmwUrUwMSEPzD3jugoY4GO8iriOMRu4
# MGcwrR+vYeFIw8dRXz9qPJsyCHRsh0Bq3xyeJKxtea/LBBTToYIDIDCCAxwGCSqG
# SIb3DQEJBjGCAw0wggMJAgEBMHcwYzELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRp
# Z2lDZXJ0LCBJbmMuMTswOQYDVQQDEzJEaWdpQ2VydCBUcnVzdGVkIEc0IFJTQTQw
# OTYgU0hBMjU2IFRpbWVTdGFtcGluZyBDQQIQDE1pckuU+jwqSj0pB4A9WjANBglg
# hkgBZQMEAgEFAKBpMBgGCSqGSIb3DQEJAzELBgkqhkiG9w0BBwEwHAYJKoZIhvcN
# AQkFMQ8XDTIyMTExMjIxNDczMVowLwYJKoZIhvcNAQkEMSIEINA+8TrRWMbSh7e4
# iWzJtQjACuqv6YalorP9l58st/pEMA0GCSqGSIb3DQEBAQUABIICAJ0BmySlWx0B
# X2h5KXVXXqwMDRWP8OfVNmQaepSxxRKom3prEAhsVCAhEaJC/imbhI+Tn7It65zv
# 2AaVK2N8ab7wZTKK6ATF8E2fPqiowMxSBhRZ2jhu7+2eJbk9/JwO2Zn/1YULKm+/
# yOfEdFMfisAV+qX+MaIbVePtXrLSLwGY0Qzx5Ia9sY9/y6u+9zFXEEnGoC92yFzC
# tDhFvgvP59lS3tAM7mz7ywNi0+/ipMY+qY5yQLLZEpIV3tPY/01b2qs8Ezj85zDt
# B8mlLzBxCtMmPiO9jRv03Uap+8rC40q1tnDFv7lcDMSiCyiQJcanj6LKuh4YI7op
# 989U6spQnf6kBN4joQHv8dD7LrNNWUMvq+pCyysVmZCK0ZYZqvufAppr1ItyCA6V
# PQS5nurVE3rffHZLGK1pApaBVkzo/K0gboFUHTB+uN4lP7Spb384GyyeRo9MLPoD
# n3x4MBVQ9YJUbbiXutxQ6H8iwAG9p5SAVGtDkrMVmEb8D1e01lq26hLdNX+J8n1f
# XHxcQ2n6JaTz1NQ+q5IPVrRmF5oSDUX9pYt1vr9VXvugU7/CQzN67g5fTSdjpwS6
# wHwu24Du9DYVXCi3C+28BMEp3AZ0+TbJLJzwZTagEj+EgV6soW55mpyXizw7ySN+
# xD/A4WA9JuxhlEeDYfILgLyTl6oM2Ta7
# SIG # End signature block
